function Main()
  TITLE="DX7-EDIT - SysEx file editor:  "
  local FS=wx.wxDEFAULT_FRAME_STYLE-wx.wxMAXIMIZE_BOX-wx.wxRESIZE_BORDER

  FRAME=wx.wxFrame(wx.NULL,-1,TITLE,wx.wxDefaultPosition,wx.wxSize(432,310),FS) -- Main GUI frame, supports menus and all good stuffs.

  PANEL=wx.wxPanel(FRAME,-1)                                                    -- Dialog-like control/surface in a Frame.

  PAGES=wx.wxNotebook(                                                          -- Tabbed set of pages to group parameter controls.
    PANEL,
    -1,
    wx.wxPoint(11,30),
    wx.wxSize(405,230),
    wx.wxNB_TOP
  )

  AT,BT,CT,DT,MT,VT={},{},{},{},{0,0,0,0,0,0},{}                                -- ID-indexed tables: Actuators, Bank, Controls, Data, Muting, Voices.
  OCB,ASW,HLP={},-1,-1                                                          -- Operator Copy Buffer. Algorithm and Help display init.
  MIDI="Buffers/MIDI OUT"                                                       -- Path to MIDI Buffer file, for viewing and transmission.
  EDIT="Buffers/Edit"                                                           -- Path to Edit Buffer file, for temp save. Survives shutdown.
  COMP="Buffers/Compare"                                                        -- Path to Compare Buffer file, used in voice compare mode.
  RCLL="Buffers/Recall"                                                         -- Path to Recall Buffer file, which holds last edit session.
  INIT="Buffers/Init"                                                           -- Path to INIT VOICE file, which holds default parameters.
  CONF="Config"                                                                 -- Path to the Config file, saves a few things between sessions.

  function Config(WF)
    local F=io.open(CONF)  if F then  F=io.close(F)  end                        -- Test if Config file exists. If it does, run it as Lua code.
    if (F and not WF) then  dofile(CONF)  end                                   -- WF=Write Flag. If set, do not read Config file first.
    local C=""                                                                  -- Always rebuild Config if called, with known values or defaults.
    .."BASE_DIR=\""..(string.gsub(BASE_DIR or "","\\","/")).."\"\n"             -- BASE_DIR is the working directory for Load/Save dialogs.
    .."FILENAME=\""..(FILENAME or "").."\"\n"                                   -- FILENAME is the last accessed file for Load/Save dialogs.
    .."PG_N="..(PG_N or 0).."\n"                                                -- PG_N is the last accessed controls page.
    .."EF="..(EF or 0).."\n"                                                    -- EF is the Edit Flag; edit sessions are retained at shutdown.
    .."EDM="..(EDM or 0).."\n"                                                  -- EDM is the Envelope Display Mode, RLRLRLRL or RRRRLLLL.
    .."DEV="..(DEV or 1).."\n"                                                  -- DEV is the Device number, which addresses remote machines.
    F=io.open(CONF,"w+b")  F:write(C)  F:close()  dofile(CONF)                  -- Write file, then run it as it may have been absent at startup.
  end

  Config()
  dofile("Modules/Tools")                                                       -- Files to set up menus and pages of parameter controls.
  dofile("Modules/Page1")
  dofile("Modules/Page2")
  dofile("Modules/Page3")
  dofile("Modules/Page4")
  dofile("Modules/State")
  dofile("Modules/Dialogs")
  dofile("Modules/Help")

  KSS={"OFF","ON"}                                                              -- Table for Controls display of Key Sync Switch states.
  TMS={"Ratio","Fixed"}                                                         -- Table for Controls display of Tuning Mode Switches.
  LSC={"-LIN","-EXP","+EXP","+LIN"}                                             -- Table for Controls display of Level Scaling Curves.
  KEY={"A","A#","B","C","C#","D","D#","E","F","F#","G","G#"}                    -- Table for Controls display of Key names.
  LFO={                                                                         -- Table for Controls display of LFO waveforms.
    "Triangle Wave","Sawtooth Down","Sawtooth Up",
    "Square Wave","Sine Wave","Sample/Hold"
  }
  FFM={                                                                         -- Table for Controls display of Fixed Frequency Mode values.
    1000,1023,1046,1072,1096,1122,1148,1175,1202,1230,                          -- Note! Try to find or write a function that will make these.
    1259,1288,1318,1349,1380,1413,1445,1479,1514,1549,
    1585,1622,1660,1698,1738,1778,1820,1862,1905,1950,
    1995,2042,2089,2138,2188,2239,2291,2344,2399,2455,
    2512,2570,2630,2692,2716,2818,2884,2951,3020,3090,
    3162,3236,3311,3388,3567,3548,3631,3715,3802,3890,
    3981,4074,4169,4266,4365,4467,4571,4677,4786,4898,
    5012,5129,5248,5370,5495,5623,5754,5888,6026,6166,
    6310,6457,6607,6761,6918,7079,7244,7413,7586,7762,
    7943,8128,8318,8511,8718,8913,9120,9333,9550,9772
  }

  FRAME:Connect(-1,wx.wxEVT_COMMAND_NOTEBOOK_PAGE_CHANGED,                      -- When any control page is selected, keep focus on page tab.
    function(EV)  PAGES:SetFocus()  PG_N=PAGES:GetSelection()  end
  )

  FRAME:Connect(-1,wx.wxEVT_CLOSE_WINDOW,                                       -- When anything tries to close the program, save config file first.
    function(EV)  Config(1)  FRAME:Destroy()  end                               -- Must be Destroy(), as Close() would cause infinite loop.
  )

  BANK:Connect(-1,wx.wxEVT_COMMAND_BUTTON_CLICKED,                              -- When a Bank voice is selected, save Voice Number and exit dialog.
    function(EV)  VN=EV:GetId()-400  BANK:EndModal(1)  end
  )

  CNFG:Connect(-1,wx.wxEVT_COMMAND_RADIOBUTTON_SELECTED,                        -- When a Config dialog RadioButton is selected, set EDM to 0 or 1.
    function(EV)  EDM=EV:GetId()-500  end
  )

  CNFG:Connect(-1,wx.wxEVT_COMMAND_SPINCTRL_UPDATED,                            -- When the Config DEVC SpinCtrl is selected, set Device Number.
    function(EV)  DEV=DEVC:GetValue()-1  end
  )

  CNFG:Connect(-1,wx.wxEVT_COMMAND_TEXT_ENTER,                                  -- When the Config DEVC SpinCtrl has text entry, set Device Number.
    function(EV)  DEV=DEVC:GetValue()-1  end
  )

  CNFG:Connect(-1,wx.wxEVT_COMMAND_BUTTON_CLICKED,                              -- When Config dialog is closed, save if save button is clicked.
    function(EV)  local X
      if EV:GetId()==wx.wxID_OK then  X=1  end
      Config(X)  FRAME:Enable(1)  CNFG:Show(nil)  FRAME:SetFocus(1)
    end
  )

  for I=7,151 do
    CT[I]:Connect(-1,wx.wxEVT_SCROLL_THUMBTRACK,ReadCtrl)                       -- When any SpinCtrl is updated, call function ReadCtrl().
    CT[I]:Connect(-1,wx.wxEVT_COMMAND_TEXT_ENTER,ReadCtrl)                      -- When any text is entered, call function ReadCtrl().
    CT[I]:Connect(wx.wxEVT_KILL_FOCUS,NoOp)                                     -- Prevents de-focus replacing text with lowest preset number value.
  end

  CT[152]:Connect(-1,wx.wxEVT_COMMAND_TEXT_ENTER,ReadCtrl)                      -- When Voice Name text is entered, call function ReadCtrl().

  for N=200,208 do
    TBAR:Connect(N,wx.wxEVT_COMMAND_MENU_SELECTED,ReadTool)
  end

  for I=1,18 do
    AT[I]:Connect(-1,wx.wxEVT_LEFT_DCLICK,                                      -- When Op button is double-clicked, toggle Operator Mute state.
      function(EV)  local ID=EV:GetId()  MT[ID]=1-MT[ID]  DataSend(1)  end
    )

    AT[I]:Connect(-1,wx.wxEVT_LEFT_UP,                                          -- When Op button is clicked in Edit Mode, copy operator data.
      function(EV)
        if (EV:ControlDown()) then  AlertMsg(0,2,"No Ctrl+Click yet","")  end
        if (EF==1) then  FMOpCopy(EV:GetId())  end  EV:Skip()
      end
    )

    AT[I]:Connect(-1,wx.wxEVT_RIGHT_DOWN,                                       -- When Op button is right clicked, set focus as for left clicks.
      function(EV)  AT[EV:GetId()+PAGES:GetSelection()*6]:SetFocus()  end
    )

    AT[I]:Connect(-1,wx.wxEVT_RIGHT_UP,                                         -- When Op button right click is released, call menu to paste data.
      function(EV)  if (EF==1 and OCB[1]) then  FMOpMenu(EV,0)  end  end
    )

    AT[I]:Connect(-1,wx.wxEVT_COMMAND_MENU_SELECTED,                            -- When Op button menu is used, paste all or partial data.
      function(EV)  ReadMenu(EV)  end
    )
  end

  if (EF==0) then  DataLoad(MIDI)  else  DataLoad(EDIT)  EF=1  end              -- Restore MIDI or Edit Buffer on load according to Edit Flag.
  PAGES:SetSelection(PG_N)  PAGES:SetFocus()                                    -- Set the selected page as remembered in Config at program close.
  FRAME:Centre()  FRAME:Show(true)
end

function FMOpMenu(EV)
  local ID,M=EV:GetId(),wx.wxMenu(),{}  OCB[0]=ID                               -- Get ID for Operator button. OCB[0] stores the button ID.
  M:Append(300,"Paste &All Op."..OCB[-1].." to Op."..ID,"")
  M:AppendSeparator()
  M:Append(301,"Paste &"..PAGES:GetPageText(PAGES:GetSelection()),"")
  M:UpdateUI()
  AT[ID]:PopupMenu(M,24,-17)
end

function FMOpCopy(ID,PF)  local O=(-ID+7)*21-15                                 -- PF=Page Flag, indexes data subset. O=Offset in Data Table.
  if (PF==0) then  for I=1,21 do  DT[I+O]=OCB[I]  FeedBack(I+O)  end            -- Paste all operator data from Operator Copy Buffer.
  elseif (PF==1) then  for I=15,21 do  DT[I+O]=OCB[I]  FeedBack(I+O)  end       -- Paste "Tuning / Output" subset of operator data.
  elseif (PF==2) then  for I=1,8 do  DT[I+O]=OCB[I]  FeedBack(I+O)  end         -- Paste "Envelope Generators" subset of operator data.
  elseif (PF==3) then  for I=9,14 do  DT[I+O]=OCB[I]  FeedBack(I+O)  end        -- Paste "Envelope Scaling" subset of operator data.
  else  for I=1,21 do  OCB[I]=DT[I+O]  end  OCB[-1]=ID                          -- If there is no known Page Flag, copy all operator data.
  end  if PF then  DataSave(EDIT,0)  end                                        -- If Page Flag exists, assume paste, and update Edit Buffer.
end

function FindTool(Z)
  Z=TBAR:FindToolForPosition(Z:GetX(),Z:GetY())                                 -- When toolbar (or tool) is clicked, look for tool where clicked.
  if Z then  Z=Z:GetId()  end  return Z                                         -- If tool was found, return it's ID, or otherwise return nil.
end

function ReadTool(EV)
  local ID=EV:GetId()                                                           -- Get the toolbar button's ID, which is how the tool is indexed.
  if (not TBAR:GetToolEnabled(ID)) then  return  end
  if (ID==200) then DataInit(EV)
  elseif (ID==201) then  FileCall(0,EV)                                         -- Open a file on demand.
  elseif (ID==202) then  FileCall(1,EV)                                         -- Save a file on demand.
  elseif (ID==203) then
    if (AlertMsg(1,2,"Are you sure?","Edit Recall:"))==wx.wxYES then            -- If confirmed, gets last edit session from Recall Buffer.
      DataLoad(RCLL)  DataSave(EDIT,0)  TBAR:EnableTool(205,1)  EF=1            -- Restore last edit session, enable Compare, set Edit Flag.
      DataSend(1)
    end
  elseif (ID==204) then  DataSend(1)                                            -- Render the voice in output buffer, and send data via MIDI.
  elseif (ID==205) then  EF=EF*-1 C_Switch()
  elseif (ID==206) then  Alg_Show()
  elseif (ID==207) then  Cfg_Show()
  elseif (ID==208) then  Hlp_Show()
  end
end

function ReadMenu(EV)
  local ID=EV:GetId()                                                           -- Get ID for menu entry.
  if (ID==300) then  FMOpCopy(OCB[0],0)                                         -- Paste from Operator Copy Buffer to all Operator data.
  elseif (ID==301) then  FMOpCopy(OCB[0],PAGES:GetSelection()+1)                -- Paste from OCB to Operator data for selected page.
  end
end

function ReadCtrl(EV)
  local ID=EV:GetId()                                                           -- Get ID for control index.
  if (ID==nil or ID<0) then  return
  elseif (EF==0) then  EF=1  TBAR:EnableTool(205,1)  end                        -- New edit: set Edit Flag and enable Compare mode.
  if (ID==152) then  NameData(ID)  return  end                                  -- If ID is for Voice Name, process the input seperately.
  DT[ID]=CT[ID]:GetValue()-CT[ID]:GetMin()  FeedBack(ID)  DataSave(EDIT,0)      -- Store value on 0-based scale, update display and Edit Buffer.
end

function FeedBack(ID)
  if type(ID)=="userdata" then  ID=ID.Id  end                                   -- ID may be a GUI-correcting event, in which case extract its ID.
  local M=CT[ID]:GetMax()                                                       -- Make a variable for max value.
  CT[ID]:SetValue(DT[ID]+CT[ID]:GetMin())                                       -- Get stored value for display to confirm what is entered.
  if (ID==143 or ID==148) then  CT[ID]:SetValue(KSS[DT[ID]+1])                  -- If updating Key Sync Switches, also update text display.
  elseif (ID==149) then  CT[ID]:SetValue(LFO[DT[ID]+1])                         -- If updating LFO waveform, also update text display.
  elseif (M==3) then  CT[ID]:SetValue(LSC[DT[ID]+1])                            -- If updating Level Scaling Curve, also update text display.
  elseif (M==1 and ID<130) then                                                 -- If updating Tuning Mode Switches, also update text display.
    CT[ID]:SetValue(TMS[DT[ID]+1])  TuneShow(ID+1)
  elseif (CT[ID] and CT[ID]:GetMax()==31) then  TuneShow(ID)                    -- If updating Coarse Tuning, call function to update display.
  elseif (ID==26 or ID==47 or ID==68 or ID==89 or ID==110 or ID==131) then      -- As above, for Fine Tuning. NOTE: DO NOT use CT[6] with this code.
    TuneShow(ID-1)
  elseif (ID==141) then                                                         -- As above, for Algorithm. NOTE: Test this before the Break Points.
  elseif ((ID+6)/21==math.floor((ID+6)/21)) then  KeyScale(ID)                  -- If updating Break Points, call function to update display.
  elseif (ID==151) then  KeyScale(ID)                                           -- If updating Transposition, call function to update display.
  end
end

function KeyScale(N)
  if N==151 then
    CT[N]:SetValue(string.format("%+03d",DT[N]-24)
    .." "..KEY[math.mod(DT[N]+3,12)+1]..1+math.floor((DT[N])/12))
  else
    CT[N]:SetValue(KEY[math.mod(DT[N],12)+1]
    ..math.floor((DT[N]+9)/12)-1)
  end
end

function NameData(N)
  local S,T,M=CT[N]:GetValue(),"Name Data Error:"
  M="Voice names must not be longer than ten characters,\n"
  .."and must use character codes in the range 32 - 127."
  if (string.find(S,"[^\32-\127]") or string.len(S)>10) then
    AlertMsg(0,0,M,T)
  else
    for I=1,10 do  DT[151+I]=string.byte(S,I) or 32  end
    FRAME:SetTitle(TITLE..S)  DataSave(EDIT,0)
  end
end

function TuneShow(N)
  local TC,TF,P=DT[N] or 1,DT[N+1] or 0,2                                       -- Get local copies of Coarse/Fine Tuning, or zero if none set.
  if (DT[N-1]~=1) then                                                          -- If Frequency Mode is Fixed, do this:
    if TC==0 then  TC,P=0.5,3  end    P="%0."..P.."f"                           -- Lowest DX7 coarse tuning value is 0.5. P sets display precision.
    CT[(N-4)/21-7]:SetValue(string.format(P,TF/100*TC+TC))                      -- Update a display in CT[-1] to CT[-6], index derived from ID.
  else                                                                          -- Else do this:
    local E=3-math.mod(TC,4)  P="%0."..E.."f"                                   -- E is exponent, based on original DX7 behaviour. P sets precision.
    CT[(N-4)/21-7]:SetValue(string.format(P,FFM[TF+1]/10^E).." Hz")             -- Update a display in CT[-1] to CT[-6], index derived from ID.
  end
end

function DataInit()
  if (AlertMsg(1,2,"Are you sure?","Initialise Data:"))==wx.wxYES then          -- If confirmed, gets INIT VOICE preset from Init Buffer.
    if (EF==1) then  DataSave(RCLL,0)  end  DataLoad(INIT)                      -- If Edit Flag is set, save for recall before loading new data.
    DataSave(EDIT,0)  TBAR:EnableTool(205,nil)  EF=0  DataSend(1)               -- EF is reset, so that new load does not flush INIT to RCLL.
  end
end

function FileCall(WF,EV)  local X={"Load","Save",wx.wxOPEN,wx.wxSAVE}  VN=nil   -- WF=Write Flag. (0/1). EV=Event, to be tested for ShiftDown().
  if (WF==1 and wx.wxGetKeyState(wx.WXK_SHIFT)) then
    local M,T="      Are you sure?        (Shift held while saving)\n\n"
    .."The current data will replace all voices in a bank file.\n"
    .."This can not be undone unless you have a backup!","Initialise BANK:"
    if (AlertMsg(1,0,M,T))==wx.wxYES then  VN=0  else return  end
  end
  local Y=X[WF+1].." DX7 Voice File"                                            -- Build ToolTip prompt.
  local Z="DX7 voice files  (*.syx)|*.syx|All files  (*.*)|*.*"                 -- Set filetypes for the dialog.
  local FilePane=wx.wxFileDialog(FRAME,Y,"","",Z,X[WF+3])                       -- Set up the dialog. Y is save/load type.
  FilePane:SetDirectory(BASE_DIR)                                               -- Set last used directory in advance. "" is script's base dir.
  FilePane:SetFilename(FILENAME)                                                -- Fill in last used filename in advance.
  if (FilePane:ShowModal()==wx.wxID_OK) then                                    -- Show the dialog and respond if OK is clicked, else ignore.
    BASE_DIR=FilePane:GetDirectory()                                            -- Set last used directory memory, whether changed or not.
    FILENAME=FilePane:GetFilename()                                             -- Set last used filename memory, whether changed or not.
    FILEPATH=FilePane:GetPath()                                                 -- Getting Path avoids OS local problems with / or \ in paths.
    if (EF==1) then  DataSave(RCLL,0)  end                                      -- If Edit Flag is set, save for recall before loading new data.
    if (WF==1) then  DataSave(FILEPATH,0)  end                                  -- If Write Flag is set, save the current voice data.
    DataLoad(FILEPATH,1)  TBAR:EnableTool(205,nil)  EF=0                        -- Load data (verify after save) to reset mutes, and exit edit mode.
  end  FilePane:Destroy()
end

function FileTest(FILE)  local F,S=io.open(FILE,"rb")                           -- Attempt to open the file (in binary mode).
  if (F) then  S=F:read("*a")  F:close()                                        -- If successful, read to string and close the file.
    if (string.len(S)~=163 and string.len(S)~=4104) then  S=""                  -- Check size. 163 bytes for voice files, 4104 bytes for bank files.
      AlertMsg(0,1,"File is the wrong size for DX7 voice data.")
    elseif (string.byte(S,1)~=240 or string.byte(S,-1)~=247) then  S=""         -- Check for System Exclusive message terminator bytes (F0,F7).
      AlertMsg(0,1,"File has invalid SysEx terminator bytes. (F0/F7)")
    end  return S                                                               -- Return file as string, or blank string if there is an error.
  end
end

function DataLoad(FILE,MF)  local N,S=0,FileTest(FILE)                          -- MF=Manual Flag, for direct request. N=Temp. FileTest() loads data.
  if (not S) then  AlertMsg(0,1,FILE.." does not exist.")  return
  elseif (string.len(S)==163) then  DT={0,0,0,0,0,0}                            -- DT[1 to 6] must hold data, or table inserts and removes will fail!
    for I=7,151 do  DT[I]=string.byte(S,I)  N=N+DT[I]  FeedBack(I)  end         -- Get voice data and build sum of values, and update the controls.
    for I=152,161 do  DT[I]=string.byte(S,I)  N=N+DT[I]  end                    -- Get name data and add to sum of values.
    CheckSum(S,162,N)  N=string.sub(S,152,161)  CT[152]:SetValue(N)             -- Update the voice name control with the loaded voice name data.
    if MF then  DataSave(COMP,0)  end                                           -- If manually loaded from single voice file, save to Compare Buffer.
    MT={0,0,0,0,0,0}  DataSend(MF)  FRAME:SetTitle(TITLE..N)                    -- Reset mutes, render output file, and set voice name in title bar.
  elseif (string.len(S)==4104) then  DataBank(S)  --VN=nil                      -- Set VN=nil to enable Bank dialog on next call after verifying save.
  end
  if MF then  DEV=string.byte(S,3)  end                                         -- Set Device number from voice or bank file, if Manual Flag is set.
end

function DataSave(FILE)  local N,S=0,FileTest(FILE)                             -- N=Temp. FileTest() loads data for testing.
  if (not S or string.len(S)==163) then  S=string.char(240,67,DEV,0,1,27)       -- If FILE is absent or is single voice, make DX7 Voice sysex header.
    for I=7,161 do  S=S..string.char(DT[I])  end                                -- Build the binary string for single voice data.
  elseif (string.len(S)==4104) then  DataBank(S,1)                              -- If FILE is a 32-voice bank, pack the single voice data into it.
    S=string.char(240,67,DEV,9,32,0)  for I=1,32 do  S=S..BT[I]  end            -- Build the binary string for 32 voice packed data from Bank Table.
  else return                                                                   -- If files did not pass FileTest(), do nothing.
  end
  for I=7,string.len(S) do  N=N+string.byte(S,I)  end                           -- Collect stored values for Checksum.
  S=S..string.char(math.mod(2^32-N,128),247)                                    -- Append Checksum and Terminator. LUA NEEDS BITWISE OPERATORS!!! >:)
  local F=io.open(FILE,"w+b")  F:write(S)  F:close()
end

function DataSend(SF)                                                           -- SF=Send Flag. If set, send the rendered output file via MIDI.
  local IF,X={wx.wxColour(140,140,140)}  IF[0]=wx.wxNullColour                  -- IF=Indicator Flag, shows whether FM Operator is muted or not.
  for I=1,6 do  local X=(7-I)*21+2                                              -- X=Temp, holding offsets for Output Levels to save to MIDI Buffer.
    for N=0,2 do  AT[I+N*6]:SetForegroundColour(IF[MT[I]])  end                 -- Set mute indicators according to the Muting Table, for each page N.
    if (MT[I]==1) then  MT[I],DT[X]=DT[X],0  else  MT[I]=nil  end               -- If Mute set, copy OL to Muting Table and mute OL, else disable mute.
  end  DataSave(MIDI,0)  if SF then  wx.wxExecute("SysEx!.exe "..MIDI)  end     -- Save voice to MIDI Buffer, and send to remote machine if SF is set.
  for I=1,6 do
    if (MT[I]) then  DT[(7-I)*21+2],MT[I]=MT[I],1  else  MT[I]=0  end           -- If mute not disabled, restore OL and mute setting, else reset mute.
  end
end

function DataBank(S,WF)  BT={}  local N=0                                       -- S is raw data string. WF is Write Flag. BT is Bank Table, N is Temp.
  for I=7,3975,128 do  table.insert(BT,string.sub(S,I,I+127))                   -- Fill the empty Bank Table, 128 bytes for each packed voice.
    for J=1,128 do  N=N+string.byte(BT[table.getn(BT)],J)  end                  -- Collect the value of each byte in each voice for bank checksum.
  end  CheckSum(S,4103,N)                                                       -- Verify the checksum.
  if (VN==nil) then
    for I=1,32 do  N=string.format("%02d: ",I)                                  -- Get voice names from Bank Table, write them to selector buttons.
      VT[I]:SetLabel(N..string.sub(BT[I],-10,-1))
    end  BANK:SetTitle(" Bank:  "..FILENAME)                                    -- Set the bank file's name on the title bar of the selection dialog.
    BANK:CentreOnParent()  BANK:ShowModal(wx.TRUE)                              -- Show selector dialog. Button ID will be set to VN (Voice Number).
  end
  if (VN<433 and not WF) then  DT={0,0,0,0,0,0}                                 -- 433 is Close button ID-400. WF=Write Flag. No empty elements in DT!
    for I=1,128 do  DT[I+6]=string.byte(BT[VN],I)  end                          -- Unpack the selected 128 byte packed voice, filling the Data Table.
    for I=1,6 do  N=I*21-14                                                     -- Decompress the packed values to own bytes, expanding the table.
      table.insert(DT,N+11,math.mod(DT[N+11],4))                                -- LUA NEEDS BITWISE OPERATORS!!! >:) This really ought to be easier!
      DT[N+12]=(math.mod(DT[N+12],16)-DT[N+11])/4
      table.insert(DT,N+18,DT[N+13])
      DT[N+13]=math.mod(DT[N+13],8)  DT[N+18]=(DT[N+18]-DT[N+13])/8
      table.insert(DT,N+14,math.mod(DT[N+14],4))
      DT[N+15]=(math.mod(DT[N+15],32)-DT[N+14])/4
      table.insert(DT,N+17,math.mod(DT[N+17],2))
      DT[N+18]=(math.mod(DT[N+18],64)-DT[N+17])/2
    end  DT[141]=math.mod(DT[141],32)
    table.insert(DT,142,math.mod(DT[142],8))
    DT[143]=(math.mod(DT[143],64)-DT[142])/8
    table.insert(DT,148,math.mod(DT[148],2))  DT[149]=(DT[149]-DT[148])/2
    table.insert(DT,149,math.mod(DT[149],8))  DT[150]=(DT[150]-DT[149])/8
    DT[151]=math.mod(DT[151],64)
    DataSave(COMP,0)  DataLoad(COMP,1)                                          -- Save single voice to Compare Buffer, and reload, sending via MIDI.
  elseif (VN<433) then
    for I=1,6 do  N=I*17-10                                                     -- Compress the grouped values into single bytes, contracting the table.
      DT[N+11]=DT[N+11]+DT[N+12]*4  table.remove(DT,N+12)
      DT[N+12]=DT[N+12]+DT[N+19]*8  table.remove(DT,N+19)
      DT[N+13]=DT[N+13]+DT[N+14]*4  table.remove(DT,N+14)
      DT[N+15]=DT[N+15]+DT[N+16]*2  table.remove(DT,N+16)
    end
    DT[118]=DT[118]+DT[119]*8  table.remove(DT,119)
    DT[124]=DT[124]+DT[125]*8  table.remove(DT,125)
    DT[123]=DT[123]+DT[124]*2  table.remove(DT,124)
    S=""  for I=1,128 do  S=S..string.char(DT[I+6])  end                        -- Replace selected 128 byte packed voice with new one from Data Table.
    if (VN==0) then  for I=1,32 do  BT[I]=S  VN=1  end                          -- In special case of VN=0, bank initialisation, fill all of BT.
    else  BT[VN]=S
    end
  end
end

function C_Switch()  local T=EF~=-1                                             -- Compare mode switching.
  P1:Enable(T)  P2:Enable(T)  P3:Enable(T)  P4:Enable(T)
  if T then  DataLoad(EDIT)  else  DataLoad(COMP)  end
  for N=200,208 do
    if (N~=204 and N~=205) then  TBAR:EnableTool(N,T)  end
  end
end

function CheckSum(DATA,CS,SUM)
  if (string.byte(DATA,CS)~=math.mod(2^32-SUM,128)) then                        -- Calculate checksum, and compare with stored checksum value.
    AlertMsg(0,0,"File data is corrupt. (Invalid checksum.)")                   -- Alerts if checksum is wrong, but allows the data to be used.
  end
end

function AlertMsg(YN,ICON,M,T)  local F="File Error:"                           -- General alert. "File Error:" is default title. If YN=1, use Yes/No buttons.
  return wx.wxMessageBox(M,T or F,wx.wxOK+(YN or 0)*6+256*2^ICON,FRAME)         -- ICON=0/1/2/3=Exclamation/Error/Question/Information. M=Message. T=Title.
end

function Cfg_Show()
  DEVC:SetValue(DEV+1)  EDMC[EDM]:SetValue(1)
  CNFG:CentreOnParent()  CNFG:Show(1)  FRAME:Enable(nil)  CNFG_CA:SetFocus()
end

function Alg_Show()  ASW=-ASW
  local T,Y=ASW~=1,math.mod(DT[141],8)  X=Y*53+3  Y=(DT[141]-Y)/8*60+39         -- T is Toggle, 1/nil. X,Y plot 2D H-scan, Coordinate*Scale+Offset.
  if (ASW==1) then
    ALG=wx.wxBitmap("Bitmaps/ALGS.bmp",wx.wxBITMAP_TYPE_BMP)                    -- Get images for Algorithm Chart and Selection Indicator display.
    IND=wx.wxBitmap("Bitmaps/LINE.bmp",wx.wxBITMAP_TYPE_BMP)
    ALG=wx.wxStaticBitmap(PANEL,-1,ALG,wx.wxPoint(1,25),wx.wxSize(424,240))     -- Display images as directed. The indicator overlays the chart.
    IND=wx.wxStaticBitmap(PANEL,-1,IND,wx.wxPoint(X,Y),wx.wxSize(18,1))
  else  ALG:Destroy()  IND:Destroy()
  end  PAGES:Show(T)                                                            -- When showing the Algorithm dialog, hide the edit controls.
  for N=200,208 do
    if ((N~=205 or EF==1) and N~=206) then  TBAR:EnableTool(N,T)  end
  end
end

function Hlp_Show()  HLP=-HLP  local T=HLP~=1                                   -- T is Toggle, 1/nil. When showing Help, hide the edit controls.
  HELP:Show(not T)  PAGES:Show(T)
  for N=200,208 do
    if ((N~=205 or EF==1) and N~=208) then  TBAR:EnableTool(N,T)  end
  end
end

function NoOp()  end                                                            -- Does nothing! Called to prevent unwanted inbuilt event handling.

--========================================================================

Main()
